<?php

namespace Informax\ImaxMultiAlmacen;

use Context;
use PrestaShopBundle\Api\QueryParamsCollection;
use PrestaShopBundle\Api\Stock\Movement;
use PrestaShopBundle\Api\Stock\MovementsCollection;
use StockAvailable;
use PrestaShopBundle\Entity\ProductIdentity;
use Configuration;

class StockRepository extends \PrestaShopBundle\Entity\Repository\StockRepository {
    private $decoratedService;

    /**
     * StockRepository constructor.
     *
     * @param DecoratedService $decoratedService
     * @param ContainerInterface $container
     * @param Connection $connection
     * @param EntityManager $entityManager
     * @param ContextAdapter $contextAdapter
     * @param ImageManager $imageManager
     * @param StockManager $stockManager
     * @param $tablePrefix
     */
    public function __construct($decoratedService, $container, $connection, $entityManager, $contextAdapter, $imageManager, $stockManager, $tablePrefix) {
        parent::__construct($container, $connection, $entityManager, $contextAdapter, $imageManager, $stockManager, $tablePrefix);
        $this->decoratedService = $decoratedService;
    }
    
    /**
     * Sirve para llamar al objeto original en las funciones no decoradas.
     * @param string $name
     * @param array $arguments
     * @return mixed
     */
    public function __call($name, $arguments) {
        return call_user_func_array(array($this->decoratedService, $name), $arguments);
    }
    
    /**
     * @return string
     */
    private function orderByProductIds() {
        return 'ORDER BY p.id_product DESC, COALESCE(pa.id_product_attribute, 0) DESC';
    }
    
    /**
     * @param MovementsCollection $movements
     *
     * @return array
     */
    public function bulkUpdateStock(MovementsCollection $movements) {
        $products = $movements->map(function (Movement $movement) {
            return $this->updateStock($movement);
        });

        return $products;
    }

    /**
     * @param Movement $movement
     * @param bool $syncStock
     *
     * @return mixed
     */
    public function updateStock(Movement $movement, $syncStock = true) {
        $productIdentity = $movement->getProductIdentity();
        $delta = $movement->getDelta();

        if ($productIdentity->getProductId() && $delta !== 0) {
            $id_warehouse = (int)Configuration::getGlobalValue('imax_multi_alma_sell_stock_warehouse');
            if(!$id_warehouse) {
                $id_warehouse = Context::getContext()->warehouse->id;
            }
            StockAvailable::updateQuantity($productIdentity->getProductId(), $productIdentity->getCombinationId(), $delta, null, false, array(), $id_warehouse);
        }

        return $this->selectStockBy($productIdentity);
    }
    
    /**
     * @param ProductIdentity $productIdentity
     *
     * @return mixed
     */
    private function selectStockBy(ProductIdentity $productIdentity) {
        $andWhereClause = '
            AND p.id_product = :product_id
            AND COALESCE(pa.id_product_attribute, 0) = :combination_id';
        $query = $this->selectSql($andWhereClause);

        $statement = $this->connection->prepare($query);
        $this->bindStockManagementValues($statement, null, $productIdentity);

        $statement->execute();
        $rows = $statement->fetchAll();
        $statement->closeCursor();
        $this->foundRows = $this->getFoundRows();

        if (count($rows) === 0) {
            throw new ProductNotFoundException(
                sprintf(
                    'Product with id %d and combination id %d can not be found',
                    $productIdentity->getProductId(),
                    $productIdentity->getCombinationId()
                )
            );
        }

        $rows = $this->addAdditionalData($rows);

        return $this->castNumericToInt($rows)[0];
    }
  
    /**
     * @param QueryParamsCollection $queryParams
     *
     * @return mixed
     */
    public function getData(QueryParamsCollection $queryParams) {
        $query = $this->selectSql(
                $this->andWhere($queryParams),
                $this->having($queryParams),
                $this->orderBy($queryParams)
            ) . $this->paginate();

        $statement = $this->connection->prepare($query);
        $this->bindStockManagementValues($statement, $queryParams);

        $statement->execute();
        $rows = $statement->fetchAll();
        $statement->closeCursor();
        $this->foundRows = $this->getFoundRows();

        $rows = $this->addAdditionalData($rows);

        return $this->castNumericToInt($rows);
    }
    
    /**
     * @param string $andWhereClause
     * @param string $having
     * @param null $orderByClause
     *
     * @return mixed
     */
    protected function selectSql(
        $andWhereClause = '',
        $having = '',
        $orderByClause = null
    ) {
        if (null === $orderByClause) {
            $orderByClause = $this->orderByProductIds();
        }

        $combinationNameQuery = $this->getCombinationNameSubquery();
        $id_warehouse = (int)Configuration::getGlobalValue('imax_multi_alma_sell_stock_warehouse');
        if(!$id_warehouse) {
            $id_warehouse = Context::getContext()->warehouse->id;
        }
 
        return str_replace(
            array(
                '{and_where}',
                '{having}',
                '{order_by}',
                '{table_prefix}',
                '{combination_name}',
                '{id_warehouse}'
            ),
            array(
                $andWhereClause,
                $having,
                $orderByClause,
                $this->tablePrefix,
                $combinationNameQuery,
                $id_warehouse
            ),
            'SELECT SQL_CALC_FOUND_ROWS
          p.id_product                                                                      AS product_id,
          COALESCE(pa.id_product_attribute, 0)                                              AS combination_id,
          IF(COALESCE(p.reference, "") = "", "N/A", p.reference)                            AS product_reference,
          IF(COALESCE(pa.reference, "") = "", "N/A", pa.reference)                          AS combination_reference,
          pl.name                                                                           AS product_name,
          p.id_supplier                                                                     AS supplier_id,
          COALESCE(s.name, "N/A")                                                           AS supplier_name,
          COALESCE(ic.id_image, 0)                                                          AS product_cover_id,
          p.active,
          sw.usable_quantity                                                                AS product_available_quantity,
          sw.physical_quantity                                                              AS product_physical_quantity,
          0                                                                                 AS product_reserved_quantity,
          IF(COALESCE(pa.id_product_attribute, 0) > 0, COALESCE(pas.low_stock_threshold, "N/A"),
             COALESCE(ps.low_stock_threshold,
                      "N/A"))                                                               AS product_low_stock_threshold,
          IF(COALESCE(pa.id_product_attribute, 0) > 0, IF(sw.usable_quantity <= pas.low_stock_threshold, 1, 0),
             IF(sw.usable_quantity <= ps.low_stock_threshold, 1, 0))                               AS product_low_stock_alert,
          {combination_name}
        FROM {table_prefix}product p
          LEFT JOIN {table_prefix}product_attribute pa ON (p.id_product = pa.id_product)
          LEFT JOIN {table_prefix}product_lang pl ON (p.id_product = pl.id_product AND pl.id_lang = :language_id)
          INNER JOIN {table_prefix}product_shop ps ON (p.id_product = ps.id_product AND ps.id_shop = :shop_id)
          LEFT JOIN {table_prefix}stock sw
            ON (p.id_product = sw.id_product AND sw.id_product_attribute = COALESCE(pa.id_product_attribute, 0)
             AND sw.id_warehouse = {id_warehouse})
          LEFT JOIN {table_prefix}image ic ON (p.id_product = ic.id_product AND ic.cover = 1)
          LEFT JOIN {table_prefix}image_shop ims
            ON (p.id_product = ims.id_product AND ic.id_image = ims.id_image AND ims.id_shop = :shop_id AND ims.cover = 1)
          LEFT JOIN {table_prefix}supplier s ON (p.id_supplier = s.id_supplier)
          LEFT JOIN {table_prefix}product_attribute_combination pac ON (pac.id_product_attribute = pa.id_product_attribute)
          LEFT JOIN {table_prefix}product_attribute_shop pas
            ON (pas.id_product = pa.id_product AND pas.id_product_attribute = pa.id_product_attribute AND
                pas.id_shop = :shop_id)                     
        WHERE
          p.state = :state
          {and_where}
        GROUP BY p.id_product, pa.id_product_attribute
        HAVING 1 {having}
        {order_by}
        '
        );
    }
}
