<?php

namespace Informax\ImaxMultiAlmacen;

use Doctrine\DBAL\Connection;
use PrestaShop\PrestaShop\Adapter\Product\Stock\Validate\StockAvailableValidator;
use PrestaShop\PrestaShop\Adapter\Shop\Repository\ShopGroupRepository;
use PrestaShop\PrestaShop\Adapter\Product\Stock\Repository\StockAvailableRepository as StockAvailableRepositoryPresta;
use PrestaShop\PrestaShop\Core\Domain\OrderState\ValueObject\OrderStateId;
use PrestaShop\PrestaShop\Core\Domain\Product\Combination\ValueObject\CombinationId;
use PrestaShop\PrestaShop\Core\Domain\Product\Stock\Exception\StockAvailableNotFoundException;
use PrestaShop\PrestaShop\Core\Domain\Product\Stock\ValueObject\StockId;
use PrestaShop\PrestaShop\Core\Domain\Product\ValueObject\ProductId;
use PrestaShop\PrestaShop\Core\Domain\Shop\ValueObject\ShopId;
use PrestaShop\PrestaShop\Core\Exception\CoreException;
use StockAvailable;

class StockAvailableRepository extends StockAvailableRepositoryPresta
{
    private $stockAvailableRepository, $connection, $dbPrefix;
    
    public function __construct(
        StockAvailableRepositoryPresta $stockAvailableRepository, 
        Connection $connection,
        string $dbPrefix,
        StockAvailableValidator $stockAvailableValidator,
        ShopGroupRepository $shopGroupRepository = null
    ) {
        parent::__construct($connection, $dbPrefix, $stockAvailableValidator, $shopGroupRepository);
        $this->stockAvailableRepository = $stockAvailableRepository;
        $this->connection = $connection;
        $this->dbPrefix = $dbPrefix;
    }

    /**
     * @param StockId $stockId
     *
     * @return StockAvailable
     *
     * @throws CoreException
     * @throws StockAvailableNotFoundException
     */
    public function get(StockId $stockId): StockAvailable
    {
        $stockAvailable = $this->stockAvailableRepository->get($stockId);
        $stockAvailable->quantity = \StockAvailable::getQuantityAvailableByProduct($stockAvailable->id_product, 
                $stockAvailable->id_product_attribute, $stockAvailable->id_shop);
        
        return $stockAvailable;
    }

    /**
     * @param ProductId $productId
     * @param ShopId $shopId
     *
     * @return StockAvailable
     *
     * @throws CoreException
     * @throws StockAvailableNotFoundException
     */
    public function getForProduct(ProductId $productId, ShopId $shopId = null): StockAvailable
    {
        $stockId = $this->stockAvailableRepository->getStockIdByProduct($productId, $shopId);

        return $this->get($stockId);
    }

    /**
     * @param ProductId $productId
     *
     * @throws CoreException
     */
    public function delete(ProductId $productId, ShopId $shopId = null): void
    {
        $this->stockAvailableRepository->delete($productId, $shopId);
        //TODO: En multitienda fallará.
        $this->connection->createQueryBuilder()->delete('stock')->where('id_product = :productId')->setParameter('productId', $productId)->execute();
    }

    /**
     * @param CombinationId $combinationId
     *
     * @return StockAvailable
     *
     * @throws CoreException
     * @throws StockAvailableNotFoundException
     */
    public function getForCombination(CombinationId $combinationId, ShopId $shopId = null): StockAvailable
    {
        $stockId = $this->stockAvailableRepository->getStockIdByCombination($combinationId, $shopId);

        return $this->get($stockId);
    }
    
    /**
     * Updates the physical_quantity and reserved_quantity columns for the specified Stock. Most of this function logic comes from
     * StockManager::updatePhysicalProductQuantity
     *
     * @param StockId $stockId
     * @param OrderStateId $errorStateId
     * @param OrderStateId $canceledStateId
     */
    public function updatePhysicalProductQuantity(StockId $stockId, OrderStateId $errorStateId, OrderStateId $canceledStateId): void
    {

    }

}
