/**
 * Clase para manejar la visualización de las asignaciones.
 * @type AsignacionesFormaPago
 */
class AsignacionesFormaPago {
    /**
     * 
     * @param {Array} asignaciones
     * @param {Object} traducciones
     * @param {Array} formasPagoDisponibles
     * @param {Array} gruposDisponibles
     * @param {Array} almacenesDisponibles
     * @param {String} contenedor
     * @returns {AsignacionesFormaPago}
     */
    constructor(asignaciones, traducciones, formasPagoDisponibles, gruposDisponibles, almacenesDisponibles, contenedor) {
        this.asignaciones = asignaciones;
        this.traducciones = traducciones;
        this.formasPagoDisponibles = formasPagoDisponibles;
        this.gruposDisponibles = gruposDisponibles;
        this.almacenesDisponibles = almacenesDisponibles;
        this.contenedor = contenedor;
        
        var esto = this;
        //Aplicamos los eventos de los botones de la tabla
        $('body').on('click', '.modoFormaPago input[name="agregarGrupos"]', function() {
            esto.abrirVentanaAsignacion.call(esto, new AsignacionFormaPago([], [], {}, traducciones, almacenesDisponibles));
        });
        $('body').on('click', '.modoFormaPago input[name="editarGrupos"]', function() {
            var asignacion = esto.buscarAsignacion($(this).parents('tr').data('id'));
            if(asignacion) {
                esto.abrirVentanaAsignacion.call(esto, asignacion);
            }
        });
        $('body').on('click', '.modoFormaPago input[name="eliminarGrupos"]', function() {
            if(confirm(esto.l('¿Está seguro de querer eliminar la asignación?')) && esto.eliminarAsignacion($(this).parents('tr').data('id'))) {
                esto.dibujar();
            }
        });
        $('body').on('change', '.modoFormaPago select[name="almacenFormaPago"]', function() {
            var asignacion = esto.buscarAsignacion($(this).parents('tr').data('id'));
            if(asignacion) {
                asignacion.almacen = $(this).val();
            }
        });
    }
    
    /**
     * Elimina una asignación por su id.
     * @param {String} id
     * @return {boolean}
     */
    eliminarAsignacion(id) {
        var pos = -1;
        for(var i in this.asignaciones) {
            if(this.asignaciones[i].id == id) {
                pos = i;
                this.asignaciones.splice(pos, 1);
                break;
            }
        }
        
        return pos > -1;
    }
    
    /**
     * Busca una asignación por su id.
     * @param {String} id
     * @returns {AsignacionFormaPago}
     */
    buscarAsignacion(id) {
        var asignacion = null;
        for(var i in this.asignaciones) {
            if(this.asignaciones[i].id == id) {
                asignacion = this.asignaciones[i];
                break;
            }
        }
        
        return asignacion;
    }
    
    /**
     * Traduce un texto.
     * @param {String} texto
     * @returns {String}
     */
    l(texto) {
        return (typeof this.traducciones[texto] === 'undefined' ? texto : this.traducciones[texto]);
    }
    
    /**
     * Muestra la tabla de asignaciones.
     */
    dibujar() {
        var html = '<table class="table modoFormaPago">';
        html += '<thead>';
        html += '<tr>';
        html += '<th>' + this.l('Grupos') + '</th>';
        html += '<th>' + this.l('Forma de pago') + '</th>';
        html += '<th>' + this.l('Almacén') + '</th>';
        html += '<td colspan="2"><input type="button" name="agregarGrupos" value="' + this.l('Agregar') + '"></td>';
        html += '</tr>';
        html += '</thead>';
        html += '<tbody>';
        this.asignaciones.forEach(function(asignacion) {
            html += asignacion.html();
        });
        html += '</tbody>';
        html += '</table>';
        
        $(this.contenedor).html(html);
    }
    
    /**
     * Abre una ventana para configurar la asignación.
     * @param {AsignacionFormaPago} asignacion
     */
    abrirVentanaAsignacion(asignacion) {
        var idsFormasPago = asignacion.formasPago.map(function(formaPago) {
            return formaPago.id;
        });
        if(idsFormasPago.length === 0 && this.formasPagoDisponibles.length > 0) {
            idsFormasPago.push(this.formasPagoDisponibles[0].id);
        }
        
        var formasPagoString = '<ul class="elementoFormasPago">';
        this.formasPagoDisponibles.forEach(function(formaPago) {
            var checked = (idsFormasPago.includes(formaPago.id) ? 'checked="checked"' : '');
            formasPagoString += '<li><label><input type="radio" name="formaPago" value="' + formaPago.id + '" ' + checked + ' data-nombre="' + formaPago.nombre + '"/> ' + formaPago.nombre + (formaPago.descripcion ? ' - ' + formaPago.descripcion : '') + '</label></li>';
        });
        formasPagoString += '</ul>';
        
        var esto = this, tagify;
        abrirDialogo(
            '<div><label class="titulo">' + this.l('Buscar grupo:') + ' <input type="text" name="buscarGrupo" value=""/></label></div>' +
            '<div class="resultadosBusquedaGrupos clearfix"></div>' + 
            '<div class="contenedorResultados col-lg-6"><input name="gruposSeleccionados" value=""/></div>' +
            '<p>' + this.l('Si no se selecciona grupo se utilizarán todos. (Tendrá menos prioridad que los grupos específicos)') + '</p>' + 
            '<p class="elementoFormasPago titulo">' + this.l('Formas de pago:') + '</p>' +
            formasPagoString +
            '<div class="clearfix"><input type="button" name="aceptar" value="' + this.l('Aceptar') + '"/> <input type="button" name="cancelar" value="' + this.l('Cancelar') + '"/></div>'
        , function() {
            //Etiquetas para los grupos
            tagify = new Tagify($('input[name="gruposSeleccionados"]')[0], {
                whitelist: esto.gruposDisponibles.map(function(grupoDisponible) { return { code: grupoDisponible.id, value: grupoDisponible.nombre }; }),
                enforceWhitelist : true,
                delimiters: null,
                dropdown: {
                    classname: "tags-look",
                    enabled: 0,
                    closeOnSelect: false,
                    maxItems: 300
                }
            });
            //Si ya tenemos grupos asignados los mostramos
            var tags = [];
            asignacion.grupos.forEach(function(grupo) {
                tags.push({ code: grupo.id, value: grupo.nombre });
            });
            tagify.addTags(tags);

            //Buscamos los grupos y los mostramos
            var ajaxBuscarGrupo = null;
            $('input[name="buscarGrupo"]').keyup(function() {
                if(ajaxBuscarGrupo) {
                    ajaxBuscarGrupo.abort();
                }

                ajaxBuscarGrupo = $.getJSON(baseUrl + 'imaxmultialmacen_ajax.php?callback=?', {
                    accion: 'buscarGrupo',
                    texto: $(this).val()
                }, function(respuesta) {
                    if(respuesta.ok) {
                        var html = [];
                        respuesta.datos.forEach(function(grupo) {
                            html.push('<div>' + grupo.id_group + '-' + grupo.name + ' <input type="button" name="agregarGrupo" value="+" data-value="' + grupo.name + '" data-code="' + grupo.id_group + '"/></div>');
                        });
                        $('.resultadosBusquedaGrupos').html(html.join(''));

                        $('input[name="agregarGrupo"]').click(function() {
                            tagify.addTags([{value: $(this).data('value'), code: $(this).data('code')}]);
                        });
                    }
                });
            });
        }, function() {
            //Actualizamos la asignación
            asignacion.formasPago = [];
            $('.ventaAsignacionFormaPago input[name="formaPago"]:checked').each(function() {
                asignacion.formasPago.push({
                    id: $(this).val(),
                    nombre: $(this).data('nombre')
                });
            });
            asignacion.grupos = [];
            tagify.value.forEach(function(elemento) {
                asignacion.grupos.push({
                    id: elemento.code,
                    nombre: elemento.value
                });
            });
            
            //Agregamos la asignación si no lo está ya
            if(!esto.buscarAsignacion(asignacion.id)) {
                esto.asignaciones.push(asignacion);
            }
            
            esto.dibujar();
        }, 'ventaAsignacionFormaPago');
    }
    
    /**
     * Devuelve los datos de las asignaciones.
     * @returns {Array}
     */
    serializar() {
        var asignaciones = [];
        this.asignaciones.forEach(function(asignacion) {
            asignaciones.push(asignacion.serializar());
        });
        return asignaciones;
    }
    
    /**
     * Asigna los datos de las asignaciones al objeto.
     * @param {Array} asignaciones
     */
    deserializar(asignaciones) {
        /**
         * Devuelve el nombre de un elemento de tipo id-nombre.
         * @param {Array} elementos
         * @param {String} id
         * @returns {String}
         */
        function buscarNombre(elementos, id) {
            var nombre = '';
            for(var i in elementos) {
                if(elementos[i].id == id) {
                    nombre = elementos[i].nombre;
                    break;
                }
            }
            
            return nombre;
        }
        
        this.asignaciones = []; 
        var esto = this;
        asignaciones.forEach(function(asignacion) {
            var formasPago = asignacion.formasPago.map(function(formaPago) { return { id: formaPago, nombre: buscarNombre(esto.formasPagoDisponibles, formaPago)}; });
            var grupos = asignacion.grupos.map(function(grupo) { return { id: grupo, nombre: buscarNombre(esto.gruposDisponibles, grupo)}; });
            esto.asignaciones.push(new AsignacionFormaPago(formasPago, grupos, asignacion.almacen, esto.traducciones, esto.almacenesDisponibles));
        });
    }
}