<?php

class FuncionesImaxMultiAlmacen {
	private $modulo;
	
	public function __construct($modulo) {
		$this->modulo = $modulo;
	}
    
    /**
     * Devuelve todas las direcciones que no sean de cliente.
     * @return array
     */
    public function cargarDirecciones() {
        return Db::getInstance()->executeS('SELECT id_address, alias FROM `'._DB_PREFIX_.'address` WHERE id_customer = 0 AND id_manufacturer = 0 
            AND id_supplier = 0 AND deleted = 0 AND active = 1');
    }
    
    /**
     * Devuelve las asignaciones de almacen ordenadas.
     * @return array
     */
    public function cargarAsignacionesAlmacenes() {
        $asignaciones = Db::getInstance()->executeS('SELECT * FROM `'._DB_PREFIX_.$this->modulo->sufijo.'asignacionAlmacen`');
        $asignacionesOrdenadas = array();
        foreach($asignaciones as $asignacion) {
            $asignacionesOrdenadas[$asignacion['id_country']][$asignacion['id_state']][$asignacion['id_group']] = $asignacion['id_warehouse'];
        }
        
        return $asignacionesOrdenadas;
    }
    
    /**
     * Guarda una asignacion de almacen.
     * @param int $id_country
     * @param int $id_state
     * @param int $id_group
     * @param int $id_warehouse
     */
    public function grabarAsignacionAlmacen($id_country, $id_state, $id_group, $id_warehouse) {
        $id_country = (int)$id_country;
        $id_state = (int)$id_state;
        $id_group = (int)$id_group;
        $id_warehouse = (int)$id_warehouse;
        
        return Db::getInstance()->execute('REPLACE INTO `'._DB_PREFIX_.$this->modulo->sufijo."asignacionAlmacen` 
            (id_country, id_state, id_group, id_warehouse) VALUES ('$id_country', '$id_state', '$id_group', '$id_warehouse')");
    }
    
    /**
     * Vacia la tabla de asignaciones.
     */
    public function eliminarAsignacionesAlmacen() {
        return Db::getInstance()->execute('TRUNCATE `'._DB_PREFIX_.$this->modulo->sufijo.'asignacionAlmacen`');
    }
    
    /**
     * Carga un asignacion de almacen.
     * @param int $id_country
     * @param int $id_state
     * @param int $id_group
     * @return int
     */
    public function cargarAsignacionAlmacen($id_country, $id_state, $id_group) {
        $id_country = (int)$id_country;
        $id_state = (int)$id_state;
        $id_group = (int)$id_group;
        
        $id_warehouse = (int)Db::getInstance()->getValue('SELECT id_warehouse FROM `'._DB_PREFIX_.$this->modulo->sufijo."asignacionAlmacen` 
            WHERE id_country = '$id_country' AND id_state = '$id_state' AND id_group = '$id_group'");
        if(!$id_warehouse) {
            $id_warehouse = (int)Db::getInstance()->getValue('SELECT id_warehouse FROM `'._DB_PREFIX_.$this->modulo->sufijo."asignacionAlmacen` 
                WHERE id_country = '$id_country' AND id_state = '$id_state' AND id_group = 0");
        }
        
        return $id_warehouse;
    }
  
    /**
     * Devuelve el id y el nombre de los grupos buscando por nombre e id.
     * @param string $texto
     * @return array
     */
    public function buscarGrupo($texto) {
        $texto = pSQL($texto);
        
        return Db::getInstance()->executeS('
            SELECT g.id_group, name FROM `'._DB_PREFIX_.'group` g 
                LEFT JOIN `'._DB_PREFIX_."group_lang` gl ON g.id_group = gl.id_group AND gl.id_lang = '{$this->modulo->idLang}'
            WHERE gl.name LIKE '%$texto%' OR g.id_group LIKE '%$texto%'");
    }
    
    /**
     * Devuelve el id y el nombre de los transportistas buscando por nombre e id.
     * @param string $texto
     * @return array
     */
    public function buscarTransportista($texto) {
        $texto = pSQL($texto);
        
        return Db::getInstance()->executeS("
            SELECT id_reference, name FROM `"._DB_PREFIX_."carrier` WHERE deleted = 0 AND active = 1 AND (name LIKE '%$texto%' OR id_carrier LIKE '%$texto%')");
    }
    
    /**
     * Devuelve los estados de un pais.
     * @param int $id_country
     * @return array
     */
    public function cargarEstados($id_country) {
        return State::getStatesByIdCountry($id_country, true);
    }
    
    /**
     * Crea una nueva direccion y devuelve el mensaje de error de haberlo.
     * @param array $datosDireccion
     * @return string
     */
    public function procesarDireccion($datosDireccion) {
        $error = '';
        if($datosDireccion) {
            try {
                $direccion = new Address((int)$datosDireccion['id']);
                $direccion->dni = '11111111H';
                foreach($datosDireccion as $nombre => $valor) {
                    $direccion->$nombre = $valor;
                }
                $direccion->save();
            }
            catch(Exception $e) {
                $error = $e->getMessage();
            }
        }
        else {
            $error = $this->modulo->l('Faltan datos.');
        }
        
        return [$error, $direccion];
    }
    
    /**
     * Devuelve el html de la plantilla de direccion.
     * @param int $id_address
     * @return string
     */
    public function plantillaDireccion($id_address) {
        Context::getContext()->smarty->assign('direccion', new Address($id_address));
        Context::getContext()->smarty->assign('paises', Country::getCountries($this->modulo->idLang, true));
        return $this->modulo->display($this->modulo->name, 'direccion.tpl');
    }
    
    /**
     * Anota la relacion de pedido con almacen.
     * @param int $id_order
     * @param int $id_warehouse
     * @return boolean
     */
    public function anotarAlmacenPedido($id_order, $id_warehouse) {
        $id_order = (int)$id_order;
        $id_warehouse = (int)$id_warehouse;
        return Db::getInstance()->execute('INSERT IGNORE INTO `'._DB_PREFIX_.$this->modulo->sufijo."warehouse_order` (id_order, id_warehouse) 
            VALUES ('$id_order', '$id_warehouse')");
    }
    
    /**
     * Devuelve el almacén que se usó para el pedido, puede no estar anotado.
     * @param int $id_order
     * @return int
     */
    public function cargarAlmacenPedido($id_order) {
        $id_order = (int)$id_order;
        return Db::getInstance()->getValue('SELECT id_warehouse FROM `'._DB_PREFIX_.$this->modulo->sufijo."warehouse_order`
            WHERE id_order = '$id_order'");
    }
    
    /**
     * Devuelve los almacenes ordenados.
     * @param boolean $ignore_shop
     * @param int $id_shop
     * @param boolean $images
     * @return array
     */
    public function getWarehouses($ignore_shop = false, $id_shop = null, $images = true) {
        if (!$ignore_shop) {
            if (null === $id_shop) {
                $id_shop = Context::getContext()->shop->id;
            }
        }

        $query = new DbQuery();
        $query->select('w.id_warehouse, w.name, w.reference');
        $query->from('warehouse', 'w');
        $query->where('deleted = 0');
        $query->orderBy('id_warehouse ASC');
        if (!$ignore_shop) {
            $query->innerJoin('warehouse_shop', 'ws', 'ws.id_warehouse = w.id_warehouse AND ws.id_shop = ' . (int) $id_shop);
        }

        $almacenes = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS($query);
        if($images) {
            //Agregamos las imagenes
            foreach($almacenes as &$almacen) {
                $almacen['image'] = (is_file(__DIR__."/images/{$almacen['id_warehouse']}.jpg") ? $this->modulo->getPathUri()."images/{$almacen['id_warehouse']}.jpg" : '');
            }
        }
        
        return $almacenes;
    }
    
    /**
     * Devuelve los datos del almacén a partir del id.
     * @param int $id_warehouse
     * @param boolean $images
     * @return array
     */
    public function getWarehouseById($id_warehouse, $images = true) {
        $query = new DbQuery();
        $query->select('w.id_warehouse, w.name, w.reference');
        $query->from('warehouse', 'w');
        $query->where('w.id_warehouse = ' . (int)$id_warehouse);
        $query->orderBy('id_warehouse ASC');

        $almacen = Db::getInstance(_PS_USE_SQL_SLAVE_)->getRow($query);
        if($images) {
            //Agregamos las imagenes
            $almacen['image'] = (is_file(__DIR__."/images/{$almacen['id_warehouse']}.jpg") ? $this->modulo->getPathUri()."images/{$almacen['id_warehouse']}.jpg" : '');
        }
        
        return $almacen;
    }
    
    /**
     * Indica si hay algún almacén visible para el stock para el cliente actual y les introduce una propiedad visible.
     * @param WarehouseShowStock[] $almacenes
     * @return boolean 
     */
    public function algunAlmacenVisibleStock($almacenes) {
        $visible = false;
        if(Validate::isLoadedObject(Context::getContext()->customer)) {
            $ids_group = Context::getContext()->customer->getGroups();
        }
        else {
            $ids_group = [Configuration::getGlobalValue('PS_GUEST_GROUP')];
        }
        
        foreach($almacenes as $almacen) {
            $almacen->visible = false;
            if($almacen->mostrarGrupos == 1 || ($almacen->mostrarGrupos == 2 && $this->enArray($ids_group, $almacen->grupos))) {
                $visible = true;
                $almacen->visible = true;
            }
        }
        
        return $visible;
    }
    
    /**
     * Indica si alguno de los elementos del array1 está en el array2.
     * @param mixed $array1
     * @param mixed $array2
     * @return boolean
     */
    private function enArray($array1, $array2) {
        $enArray = false;
        foreach($array1 as $elemento) {
            if(in_array($elemento, $array2)) {
                $enArray = true;
                break;
            }
        }
        
        return $enArray;
    }
    
    public function esPosibleCambiarFormaPago($formaPago){
        $valid = true;
        
        $warehouseId = $this->obtenerAlmacenPorFormaPago($formaPago);
        $warehouseIdActual = $this->modulo->getWarehouse()->id;

        $cart = Context::getContext()->cart;
        $products = $cart->getProducts();
        
        if($products && $warehouseId != $warehouseIdActual){
            $valid = false;
        }
        
        return $valid;
    }
    
    /**
     * Devuelve el html de la plantilla con el MENSAJE_FORMA_PAGO_ALMACEN
     * @return string
     */
    public function plantillaMensajeFormaPagoAlmacen($warehouseName) {
        $mensajeFormaPagoAlmacen = unserialize(Configuration::getGlobalValue($this->modulo->sufijo .'MENSAJE_FORMA_PAGO_ALMACEN'));
        if($mensajeFormaPagoAlmacen) {
            $textoMensajeFormaPagoAlmacen = $mensajeFormaPagoAlmacen[$this->modulo->idLang];
        }
        else {
            $textoMensajeFormaPagoAlmacen = $this->modulo->l('Tiene artículos que pertenecen al almacén %X%, si desea agregar productos de otro almacen antes finalice el pedido');
        }
        $textoMensajeFormaPagoAlmacen = str_replace("%X%", $warehouseName, $textoMensajeFormaPagoAlmacen);
        Context::getContext()->smarty->assign('mensajeFormaPagoAlmacen', $textoMensajeFormaPagoAlmacen);
        return $this->modulo->display($this->modulo->name, 'mensajeFormaPagoAlmacen.tpl');
    }
    
    /**
     * Devuelve el id del almacén por forma de pago
     * 
     * @param string $formaPago
     * @return type
     */
    public function obtenerAlmacenPorFormaPago($formaPago){
        if($formaPago) {
            $customer = Context::getContext()->customer;
            if(Validate::isLoadedObject($customer)) {
                $grupos = $customer->getGroups();
            }
            else {
                $grupos = [Configuration::getGlobalValue('PS_GUEST_GROUP')];
            }
            $asignacion = AsignacionAlmacenPago::cargar($formaPago, $grupos);
            if($asignacion) {
                $id_warehouse = $asignacion->id_warehouse;
            }
        }
        
        if(empty($id_warehouse)) {
            //El almacen general
            $id_warehouse = (int) Configuration::getGlobalValue($this->modulo->sufijo . 'ALMACEN_GENERAL');
        }
        
        return $id_warehouse;
    }
    
    public function obtenerListadoRelacionFormaPagoAlmacen(){
        $listadoRelacionFormaPagoAlmacen = [];
        
        $paymentModules = PaymentModule::getInstalledPaymentModules();
        foreach ($paymentModules as $pm) {
            $listadoRelacionFormaPagoAlmacen[$pm['name']] = (int) $this->modulo->getWarehouseByPayment($pm['name']);
        }
        return $listadoRelacionFormaPagoAlmacen;
    }
    
    /**
     * Sincroniza el stock virtual.
     * @return string
     */
    public function procesarStockVirtual() {
        if($this->comprobarProceso('sv')) {
            //Cargamos los productos a procesar
            $ultimo = (int)Configuration::getGlobalValue($this->modulo->sufijo.'ULTIMO_STOCK_VIRTUAL');
            $cantidad = (int)Configuration::getGlobalValue($this->modulo->sufijo.'CANTIDAD_STOCK_VIRTUAL');
            $lineasProcesables = Db::getInstance()->executeS('
                SELECT p.id_product, pa.id_product_attribute FROM `'._DB_PREFIX_.'product` p
                    LEFT JOIN `'._DB_PREFIX_."product_attribute` pa ON p.id_product = pa.id_product
                ORDER BY p.id_product, pa.id_product_attribute
                LIMIT $ultimo, $cantidad");
            if(!$lineasProcesables) {
                //Reiniciamos
                Configuration::updateGlobalValue($this->modulo->sufijo.'ULTIMO_STOCK_VIRTUAL', 0);
                $mensaje = $this->modulo->l('Proceso finalizado');
            }
            else {
                //Procesamos
                $cantidad = 0;
                foreach($lineasProcesables as $lineaProcesable) {
                    $this->actualizarStockVirtualProducto($lineaProcesable['id_product'], $lineaProcesable['id_product_attribute']);
                    $cantidad++;
                    Configuration::updateGlobalValue($this->modulo->sufijo.'ULTIMO_STOCK_VIRTUAL', $ultimo + $cantidad);
                }
                $primero = reset($lineasProcesables);
                $ultimo = end($lineasProcesables);
                $mensaje = $cantidad.' '.$this->modulo->l('elementos procesados') . ". (ids: {$primero['id_product']} - {$ultimo['id_product']})";
            }
            
            $this->terminarProceso('sv');
        }
        else {
            $mensaje = $this->modulo->l('Cron solapado');
        }
        
        return $mensaje;
    }
    
    /**
     * Comprueba si hay solapamiento.
     * @param string $sufijoTipo Un sufijo para crear distintos solapamientos.
     * @return boolean
     */
    private function comprobarProceso($sufijoTipo) {
        $modoSolapado = Configuration::getGlobalValue($this->modulo->sufijo . 'MODO_SOLAPADO');
        if ($modoSolapado) {
            return true;
        }

        $procesoActivo = Configuration::getGlobalValue($this->modulo->sufijo . 'PROCESO_ACTIVO_' . $sufijoTipo);
        if ($procesoActivo == 1) {
            $fechaUltimoProceso = Configuration::getGlobalValue($this->modulo->sufijo . 'ULTIMO_PROCESO_' . $sufijoTipo);
            if ($fechaUltimoProceso == '') {
                $fechaUltimoProceso = date('U', 1);
                Configuration::updateGlobalValue($this->modulo->sufijo . 'ULTIMO_PROCESO_' . $sufijoTipo, $fechaUltimoProceso);
            }
            $fechaActual = date('U');
            $fechaUltimoProcesoModificada = $fechaUltimoProceso + 20 * 60;
            if ($fechaActual > $fechaUltimoProcesoModificada) {
                Configuration::updateGlobalValue($this->modulo->sufijo . 'PROCESO_ACTIVO_' . $sufijoTipo, 0);
            }

            return false;
        } else {
            Configuration::updateGlobalValue($this->modulo->sufijo . 'PROCESO_ACTIVO_' . $sufijoTipo, 1);
            Configuration::updateGlobalValue($this->modulo->sufijo . 'ULTIMO_PROCESO_' . $sufijoTipo, date('U'));

            return true;
        }
    }

    /**
     * Marca la ejecucion como finalizada.
     * @param string $sufijoTipo
     */
    public function terminarProceso($sufijoTipo) {
        Configuration::updateGlobalValue($this->modulo->sufijo . 'PROCESO_ACTIVO_'.$sufijoTipo, 0);
    }
    
    /**
     * Actualiza todo el stock virtual del producto indicado.
     * @param int $id_product
     * @param int $id_product_attribute
     */
    public function actualizarStockVirtualProducto($id_product, $id_product_attribute) {
        ImaxMultiAlmacen::$actualizandoStock = true;
        
        $almacenesStockVirtual = (new AlmacenesVirtuales())->cargarPorProducto($id_product, $id_product_attribute);
        //Actualizamos el producto en los almacenes virtuales en los que debería estar
        foreach($almacenesStockVirtual as $almacenStockVirtual) {
            $almacenStockVirtual->actualizarStock($id_product, $id_product_attribute);
        }
        
        ImaxMultiAlmacen::$actualizandoStock = false;
    }
    
    /**
     * 
     * @param int $id_product
     * @param int $id_product_attribute
     * @return array [{nombre, almacenes => [{nombre, physical_quantity, usable_quantity, virtual}]}]
     */
    public function obtenerAlmacenesProducto($id_product, $id_product_attribute = false, $virtualesPrimero = false) {
        $almacenes = [];
        
        $id_product = (int)$id_product;
        $tieneCombinaciones = (new Product($id_product))->hasAttributes();
        if($tieneCombinaciones) {
            $combinaciones = [];
            $stocks = Db::getInstance()->executeS('
                SELECT DISTINCT s.*, !ISNULL(asv.id_warehouse) virtual FROM `' . _DB_PREFIX_ . 'stock` s
                    LEFT JOIN `' . _DB_PREFIX_ . ImaxMultiAlmacen::prefijo . "almacenStockVirtual` asv ON s.id_warehouse = asv.id_warehouse
                WHERE id_product = $id_product AND id_product_attribute != 0 AND (physical_quantity > 0 OR usable_quantity > 0) " . 
                ($virtualesPrimero ? 'ORDER BY asv.id_warehouse DESC' : ''));
            //Agrupamos por combinación
            foreach($stocks as $stock) {
                if(!$id_product_attribute){
                    $combinaciones[$stock['id_product_attribute']][] = $stock;
                }else if($id_product_attribute == $stock['id_product_attribute']){
                    $combinaciones[$stock['id_product_attribute']][] = $stock;
                }
            }
            //Procesamos los datos
            foreach($combinaciones as $id_product_attribute => $stocks) {
                $atributos = (new Combination($id_product_attribute))->getAttributesName(Context::getContext()->language->id);
                $almacenes[] = [
                    'nombre' => implode(', ', array_column($atributos, 'name')),
                    'almacenes' => $this->obtenerDatosAlmacenes($stocks, $virtualesPrimero)
                ];
            }
            
            //Ordenamos por nombre
            usort($almacenes, function($a, $b) {
                return strcasecmp($a['nombre'], $b['nombre']);
            });
        }
        else {
            $stocks = Db::getInstance()->executeS('
                SELECT DISTINCT s.* FROM `'._DB_PREFIX_.'stock` s
                    LEFT JOIN `' . _DB_PREFIX_ . ImaxMultiAlmacen::prefijo . "almacenStockVirtual` asv ON s.id_warehouse = asv.id_warehouse
                WHERE id_product = $id_product AND id_product_attribute = 0 AND (physical_quantity > 0 OR usable_quantity > 0) " .
                ($virtualesPrimero ? 'ORDER BY asv.id_warehouse DESC' : ''));
            if($stocks) {
                $almacenes[] = [
                    'nombre' => '',
                    'almacenes' => $this->obtenerDatosAlmacenes($stocks, $virtualesPrimero)
                ];
            }
        }
        
        return $almacenes;
    }
    
    /**
     * Devuelve los datos procesados para los almacenes pasados.
     * @param array $stocks
     * @param bool $virtualesPrimero
     * @return array [{nombre, physical_quantity, usable_quantity, virtual}]
     */
    private function obtenerDatosAlmacenes($stocks, $virtualesPrimero = false) {
        $datos = [];
        foreach($stocks as $stock) {
            $almacen = new Warehouse($stock['id_warehouse']);
            $datos[] = [
                'nombre' => $almacen->name,
                'physical_quantity' => $stock['physical_quantity'],
                'usable_quantity' => $stock['usable_quantity'],
                'virtual' => $stock['virtual']
            ];
        }
        
        //Ordenamos por nombre
        usort($datos, function($a, $b) use ($virtualesPrimero) {
            if($a['virtual'] == $b['virtual'] || !$virtualesPrimero) {
                $orden = strcasecmp($a['nombre'], $b['nombre']);
            }
            elseif($a['virtual']) {
                $orden = -1;
            }
            else {
                $orden = 1;
            }
            
            return $orden;
        });
        
        return $datos;
    }
    
    /**
     * Copia los datos de la tabla stock_available a la stock, truncando lo actual.
     * @param int $id_shop
     * @param int $id_warehouse
     */
    public function copiarStockAvailableEnStock($id_shop, $id_warehouse) {
        $id_shop = (int)$id_shop;
        $id_warehouse = (int)$id_warehouse;
        
        Db::getInstance()->execute('TRUNCATE TABLE `'._DB_PREFIX_.'stock`');
        Db::getInstance()->execute('
            INSERT INTO `'._DB_PREFIX_."stock` (`id_product`, `id_product_attribute`, `id_warehouse`, `reference`, `physical_quantity`, `usable_quantity`) 
            SELECT sa.`id_product`, sa.`id_product_attribute`, $id_warehouse, COALESCE(pa.`reference`, p.reference), 
                    IF(`physical_quantity` < 0, 0, `physical_quantity`), IF(sa.quantity < 0, 0, sa.quantity) FROM "._DB_PREFIX_."stock_available sa
                LEFT JOIN "._DB_PREFIX_."product p ON sa.id_product = p.id_product
                LEFT JOIN "._DB_PREFIX_."product_attribute pa ON sa.id_product = pa.id_product AND sa.id_product_attribute = pa.id_product_attribute
            WHERE id_shop = $id_shop");
        Db::getInstance()->execute('UPDATE `'._DB_PREFIX_."order_detail` SET id_warehouse = $id_warehouse");
        if(Module::isEnabled('gestavdstock')) {
            Db::getInstance()->execute('TRUNCATE `'._DB_PREFIX_.$this->modulo->sufijo.'extraMovimiento_advanced`');
            Db::getInstance()->execute('
                INSERT INTO `'._DB_PREFIX_.$this->modulo->sufijo."extraMovimiento_advanced` (id_stock_available_advanced, id_warehouse) 
                SELECT id_stock_available_advanced, $id_warehouse FROM `"._DB_PREFIX_.'stock_available_advanced`');
        }
    }
    
    /**
     * Anota la relacion de carrito con almacen.
     * @param int $id_cart
     * @param int $id_warehouse
     * @return boolean
     */
    public function anotarCarritoPedido($id_cart, $id_warehouse) {
        $id_cart = (int)$id_cart;
        $id_warehouse = (int)$id_warehouse;
        return Db::getInstance()->execute('REPLACE INTO `'._DB_PREFIX_.$this->modulo->sufijo."warehouse_cart` (id_cart, id_warehouse) 
            VALUES ('$id_cart', '$id_warehouse')");
    }
    
    /**
     * Devuelve el almacén que se usó para el carrito, puede no estar anotado.
     * @param int $id_cart
     * @return int
     */
    public function cargarAlmacenCarrito($id_cart) {
        $id_cart = (int)$id_cart;
        return Db::getInstance()->getValue('SELECT id_warehouse FROM `'._DB_PREFIX_.$this->modulo->sufijo."warehouse_cart`
            WHERE id_cart = '$id_cart'");
    }
    
    /**
     * Graba los transportistas para un almacén.
     * @param int $id_warehouse
     * @param int[] $transportistas
     * @return boolean
     */
    public function grabarTransportistasAlmacen($id_warehouse, $transportistas) {
        $id_warehouse = (int)$id_warehouse;
        $transportistasString = pSQL(serialize($transportistas), true);
        return Db::getInstance()->execute('REPLACE INTO `'._DB_PREFIX_.$this->modulo->sufijo."warehouse_carrier` (id_warehouse, transportistas) 
            VALUES ('$id_warehouse', '$transportistasString')");
    }
    
    /**
     * Devuelve los transportistas para un almacén.
     * @param int $id_warehouse
     * @return int[]
     */
    public function obtenerTransportistasAlmacen($id_warehouse) {
        $id_warehouse = (int)$id_warehouse;
        $transportistas = Db::getInstance()->getValue('SELECT transportistas FROM `'._DB_PREFIX_.$this->modulo->sufijo."warehouse_carrier`
            WHERE id_warehouse = '$id_warehouse'");
        return ($transportistas ? unserialize($transportistas) : []);
    }
    
    /**
     * Elimina las asignaciones de transportista para almacenes.
     * @return boolean
     */
    public function eliminarTransportistasAlmacen() {
        return Db::getInstance()->execute('TRUNCATE `'._DB_PREFIX_.$this->modulo->sufijo.'warehouse_carrier`');
    }
}
