/* global mensaje_errorCodigo */

/**
 * Procesa numeros de serie basados en plantilla.
 * @param {Products} productos
 * @param {int} modoPlantilla
 * @param {String} plantilla
 * @param {String} separadorPlantilla
 * @param {boolean} conservarSeparador
 * @param {String} partesPlantilla
 * @param {String} partesPlantillaLote
 * @param {Multiplicador} multiplicador
 * @param {boolean} deluxeTracking
 * @returns {ProcesaCodigoModoNumerosSerie}
 */
function ProcesaCodigoConPlantilla(productos, modoPlantilla, plantilla, separadorPlantilla, conservarSeparador, partesPlantilla, 
        partesPlantillaLote, multiplicador, deluxeTracking) {
    this.productos = productos;
    this.modoPlantilla = modoPlantilla;
    this.plantilla = plantilla;
    this.separadorPlantilla = separadorPlantilla;
    this.conservarSeparador = conservarSeparador;
    if($.trim(partesPlantilla)) {
        this.partesPlantilla = partesPlantilla.split(',');
    }
    else {
        this.partesPlantilla = [];
    }
    if($.trim(partesPlantillaLote)) {
        this.partesPlantillaLote = partesPlantillaLote.split(',');
    }
    else {
        this.partesPlantillaLote = [];
    }
    this.multiplicador = multiplicador;
    this.ultimoProducto = null;
    this.deluxeTracking = deluxeTracking;
}

/**
 * Procesa el codigo escaneado.
 * @param {String} codigoEscaneado
 */
ProcesaCodigoConPlantilla.prototype.procesar = function(codigoEscaneado) {
    var multiplicador = this.multiplicador.getMultiplicador(codigoEscaneado);
    codigoEscaneado = this.multiplicador.getCodigo(codigoEscaneado);
    
    //Extraemos el identificador
    var identificador = this.extraerIdentificador(codigoEscaneado);
    var producto = this.productos.getByIdentifier(identificador);
    if (identificador && producto && producto.usaNumeroSerie) {
        //Lectura correcta, aumentamos la cantidad y almacenamos el numero de serie
        this.ultimoProducto = producto;
        producto.cantidadActual += multiplicador;
        if(typeof producto.numerosSerie === 'undefined') {
            producto.numerosSerie = [ ];
        }
        var lote = this.extraerLote(codigoEscaneado);
        if(lote) {
            producto.numerosSerie.push(lote);
            if(this.deluxeTracking) {
                reservarNumerosSerie(producto, multiplicador, lote);
            }
        }
        actualizarCantidadTabla(producto);
    }
    else {
        //Lo hacemos probando si es el identificador
        producto = this.productos.getByIdentifier(codigoEscaneado);
        if(producto && !producto.usaNumeroSerie) {
            producto.cantidadActual += multiplicador;

            actualizarCantidadTabla(producto);
        }
        else {
            mensaje(mensaje_errorCodigo + ': ' + codigoEscaneado, 'error');
        }
    }
};

/**
 * Extrae el identificador de producto de un codigo escaneado.
 * @param {String} numeroSerie
 * @returns {String}
 */
ProcesaCodigoConPlantilla.prototype.extraerIdentificador = function(numeroSerie) {
    var identificador = '';
    
    if(this.modoPlantilla == 1) {
        //Comodines
        for (var i = 0, len = this.plantilla.length; i < len; i++) {
            if (this.plantilla[i] === '#') {
                if(typeof numeroSerie[i] === 'undefined') {
                    identificador = '';
                    break;
                }
                else {
                    identificador += numeroSerie[i];
                }
            }
        }
    }
    else if(this.modoPlantilla == 2) {
        //Partes
        identificador = this.extraerPartes(numeroSerie, this.separadorPlantilla, this.partesPlantilla, this.conservarSeparador);
    }

    return identificador;
};

/**
 * Extrae el numero de serie de producto de un codigo escaneado.
 * @param {String} numeroSerie
 * @returns {String}
 */
ProcesaCodigoConPlantilla.prototype.extraerLote = function(numeroSerie) {
    var lote = '';
    
    if(this.modoPlantilla == 1) {
        //Comodines
        lote = numeroSerie;
    }
    else if(this.modoPlantilla == 2) {
        //Partes
        lote = this.extraerPartes(numeroSerie, this.separadorPlantilla, this.partesPlantillaLote, this.conservarSeparador);
    }

    return lote;
};

/**
 * Divide el codigo escaneado en partes.
 * @param {String} numeroSerie
 * @param {String} separadorPlantilla
 * @param {String} partesPlantilla
 * @param {String} conservarSeparador
 * @param {int} selectorGrupos 0 -> los grupos indicados, 1 -> el ultimo
 * @returns {String}
 */
ProcesaCodigoConPlantilla.prototype.extraerPartes = function(numeroSerie, separadorPlantilla, partesPlantilla, conservarSeparador, selectorGrupos) {
    var resultado = '';
    
    var partes = numeroSerie.split(separadorPlantilla);
    if(partesPlantilla.length || (typeof selectorGrupos !== 'undefined' && selectorGrupos == 1)) {
        if(typeof selectorGrupos !== 'undefined' && selectorGrupos == 1) {
            //Cogemos el ultimo
            resultado = partes[partes.length - 1];
        }
        else {
            for (var i = 0, len = partesPlantilla.length; i < len; i++) {
                if(typeof partes[partesPlantilla[i] - 1] === 'undefined') {
                    resultado = '';
                    break;
                }
                else {
                    resultado += partes[partesPlantilla[i] - 1] + (conservarSeparador ? separadorPlantilla : '');
                }
            }

            if(conservarSeparador && resultado.length > 0) {
                resultado = resultado.substr(0, resultado.length - 1);
            }
        }
    }
    else if(!conservarSeparador) {
        resultado = partes.join('');
    }
    else {
        resultado = numeroSerie;
    }

    return resultado;
};

/**
 * Elimina la cantidad indicada de ultimos lotes y de producto.
 * @param {int} cantidad
 * @return {boolean} 
 */
ProcesaCodigoConPlantilla.prototype.eliminarUltimosCodigos = function(cantidad) {
    var resultado = false;
    
    if(this.ultimoProducto !== null && this.ultimoProducto.cantidadActual !== null) {
        //Numeros de serie
        if(cantidad <= this.ultimoProducto.numerosSerie.length) {
            this.ultimoProducto.numerosSerie.length -= cantidad;
        }
        else {
            this.ultimoProducto.numerosSerie.length = 0;
        }

        //Cantidad de producto
        this.ultimoProducto.cantidadActual -= cantidad;
        if(this.ultimoProducto.cantidadActual < 0) {
            this.ultimoProducto.cantidadActual = 0;
        }
        
        resultado = true;
    }
    
    return resultado;
};