<?php

/**
 * Clase que facilita el uso del imaxForm.
 */
class ImaxFormWrapper {
    private $imaxForm;
    
    /**
     * 
     * @param imaxForm $imaxForm
     */
    public function __construct($imaxForm) {
        $this->imaxForm = $imaxForm;
    }
    
    /**
     * {@inheritDoc}
     */
    public function __call($name, $arguments) {
        $result = call_user_func_array([$this->imaxForm, $name], $arguments);
        if(is_a($result, 'imaxForm')) {
            return $this;
        }
        else {
            return $result;
        }
    }

    /**
     * Agrega html al formulario.
     * @param string $html
     * @return $this
     */
    public function addToForm($html) {
        $this->imaxForm->addToForm($html);
        
        return $this;
    }
    
    /**
     * Devuelve el html del formulario.
     * @param boolean $vaciarFormulario Si es true se vacia el form despues de devolver el html.
     * @return string
     */
    public function renderForm($vaciarFormulario = false) {
        return $this->imaxForm->renderForm($vaciarFormulario);
    }
    
    /**
     * Crea un elemento informativo.
     * @param string $text Texto a mostrar
     * @param string $class Clase del elemento (warning (defecto), confirm)
     * @return $this
     */
    public function createInformationText($text, $class = 'warning') {
        $this->imaxForm->createFormInfomationText($text, $class);
        
        return $this;
    }
    
    /**
     * Crea una lista de checkbox.
     * @param string $name
     * @param string $text
     * @param array $checks La clave es el valor del check y el valor es el texto.
     * @param string[] $selectedValues
     * @param string $title
     * @param array $attributes El indice es el nombre del atribute y el valor el valor.
     * @return $this
     */
    public function createCheckboxList($name, $text, $checks, $selectedValues, $title = '', $attributes = []) {
        $modifiedChecks = [];
        foreach($checks as $id => $textContent) {
            $modifiedChecks[] = [
                'value' => $id,
                'text' => $textContent
            ];
        }
        $this->imaxForm->createFormCheckboxGroupList($name, $text, $title, $modifiedChecks, $selectedValues, $attributes);
        
        return $this;
    }
    
    /**
     * Crea un checkbox.
     * @param string $name Nombre del checkbox.
     * @param string $text Texto informativo.
     * @param boolean $checked Marcado(true) o no(false).
     * @param string $title Titulo.
     * @param string $value El valor del checkbox.
     * @return $this
     */
    public function createCheckbox($name, $text, $checked, $title = '', $value = 1) {
        $this->imaxForm->createFormCheckboxGroup($name, $text, $checked, $title, $value);
        
        return $this;
    }
    
    /**
     * Crea un radio.
     * @param array $datos [{checked, value, name, text}, ...]
     * @param string $text 
     * @param array $attributes El indice es el nombre del atribute y el valor el valor.
     * @return $this
     */
    public function createRadio($datos, $text, $attributes = []) {
        $this->imaxForm->createFormRadioButtonGroup($datos, $text, $attributes);
        
        return $this;
    }
    
    /**
     * Crea un textarea.
     * @param string $name
     * @param string $text
     * @param string $value
     * @param boolean $multiLang
     * @param boolean $wysiwyg
     * @param boolean $array Indica si en caso de ser multilang el name debe crearse como array o como string.
     * @param string $title
     * @return $this
     */
    public function createTextArea($name, $text, $value, $multiLang = false, $wysiwyg = true, $array = false, $title = '') {
        $this->imaxForm->generarTextArea($name, $value, $text, $title, $multiLang, $wysiwyg, $array);
        
        return $this;
    }

    /**
     * Crea un input text.
     * @param string $name
     * @param string $text
     * @param string $value
     * @param boolean $multiLang
     * @param array $attributes El indice es el nombre del atribute y el valor el valor.
     * @param boolean $autofocus
     * @param string $title
     * @return $this
     */
    public function createText($name, $text, $value, $multiLang = false, $attributes = [], $autofocus = false, $title = '') {
        $this->imaxForm->createFormTextGroup($name, $value, $text, $title, $multiLang, false, $attributes, $autofocus);
        
        return $this;
    }
    
    /**
     * Crea un input password.
     * @param string $name
     * @param string $text
     * @param string $value
     * @param array $attributes
     * @param boolean $autofocus
     * @param string $title
     * @return $this
     */
    public function createPassword($name, $text, $value, $attributes = [], $autofocus = false, $title = '') {
        $this->imaxForm->createFormTextGroup($name, $value, $text, $title, false, true, $attributes, $autofocus);
        
        return $this;
    }
    
    /**
     * Crea un select con numeros.
     * @param string $name
     * @param string $text
     * @param int $selectedValue
     * @param int $min
     * @param int $max
     * @param int $step
     * @param string $title
     * @return $this
     */
    public function createSelectNumeric($name, $text, $selectedValue, $min, $max, $step = 1, $title = '') {
        $this->imaxForm->createFormSelectNumerico($name, $text, $selectedValue, $min, $max, $step, $title);
        
        return $this;
    }
    
    /**
     * Crea un input hidden.
     * @param string $name
     * @param string $value
     * @return $this
     */
    public function createHidden($name, $value) {
        $this->imaxForm->createHidden($name, $value);
        
        return $this;
    }
    
    /**
     * Crea un boton de submit.
     * @param string $name
     * @param string $text
     * @param array $attributes El indice es el nombre del atribute y el valor el valor.
     * @return $this
     */
    public function createSubmitButton($name, $text, $attributes = []) {
        $this->imaxForm->createSubmitButton($name, $text, $attributes);
        
        return $this;
    }
    
    /**
     * Crea un boton.
     * @param string $name
     * @param string $text
     * @param array $attributes
     * @param string $type
     * @param string $icon
     * @return $this
     */
    public function createButton($name, $text, $attributes = [], $type = 'button', $icon = 'plus') {
        $this->imaxForm->createButton($name, $text, $attributes, $type, $icon);
        
        return $this;
    }
    
    /**
     * Crea un select.
     * @param string $name
     * @param array $text 
     * @param string $data valor => visible
     * @param string $selectedValue
     * @param array $attributes El indice es el nombre del atribute y el valor el valor.
     * @param string $title
     * @return $this
     */
    public function createSelect($name, $text, $data, $selectedValue, $attributes = [], $title = '') {
        $this->imaxForm->createFormSelect($name, $text, $data, $selectedValue, $title, $attributes);
        
        return $this;
    }
    
    /**
     * {@inheritDoc}
     */
    public function __toString() {
        return $this->imaxForm->__toString();
    }
    
    /**
     * Genera un select a partir de un array asociativo.
     * @param string $name
     * @param string $text
     * @param array $data
     * @param string $selectedValue
     * @param string $keyId
     * @param string $keyValue
     * @param array $attributes
     * @param string $title
     * @return $this
     */
    public function createFormSelectFromArray($name, $text, $data, $selectedValue, $keyId, $keyValue, $attributes = [], $title = '') {
        $this->imaxForm->createFormSelectFromArray($name, $text, $data, $keyId, $keyValue, $selectedValue, $title, $attributes);
        
        return $this;
    }
    
    /**
     * Crea una lista de checkbox a partir de un array.
     * @param string $name
     * @param string $text
     * @param array $data
     * @param array $selectedValues
     * @param string $keyId
     * @param string $keyValue
     * @param int $columns La cantidad de columnas en la que se dividira la lista.
     * @param boolean $vertical
     * @param array $attributes
     * @param string $title
     */
    public function createCheckboxListFromArray($name, $text, $data, $selectedValues, $keyId, $keyValue, $columns = 3, $vertical = true, 
            $attributes = [], $title = '') {
        $this->imaxForm->createCheckboxesFromArray($name, $text, $title, $data, $keyId, $keyValue, $selectedValues, $columns, $attributes, 
                $vertical);
        
        return $this;
    }
    
    /**
     * Crea una tabla para trabajar con un ObjectModel.
     * Se puede usar un campo "overrideType" en las columnas del ObjectModel para usar tipos personalizados, ej: select (Este tipo hace 
     * necesaria una función estática "getSelectValues" que devuelva los valores del select).
     * @param string $objectModelName
     * @param string[] $columnNames Las traducciones para los nombres de las columnas.
     * @param string[] $columnOrder El orden de las columnas.
     * @param string[] $hideColumns Columnas que no se desean en la tabla.
     * @param string $aggregationColumn Nombre de la columna por la que se agrupa.
     * @param string $aggregationColumnValue Valor de la columna anterior para agrupar.
     * @param boolean $showId Mostrar el id en la tabla.
     * @return $this
     */
    public function createFormObjectModelTable($objectModelName, $columnNames = array(), $columnOrder = array(), $hideColumns = array(), 
            $aggregationColumn = '', $aggregationColumnValue = '', $showId = false) {
        $this->imaxForm->createFormObjectModelTable($objectModelName, $columnNames, $columnOrder, $hideColumns, $aggregationColumn, 
                $aggregationColumnValue, $showId);
        
        return $this;
    }
    
    /**
     * Genera una fila de datos para la tabla de ObjectModel.
     * Se puede usar un campo "overrideType" en las columnas del ObjectModel para usar tipos personalizados, ej: select (Este tipo hace 
     * necesaria una función estática "getSelectValues" que devuelva los valores del select).
     * @param string $objectModelName
     * @param mixed $id
     * @param array $fields
     * @param ObjectModel $object
     * @param string[] $hideColumns
     * @param string $aggregationColumn Nombre de la columna por la que se agrupa.
     * @param string $aggregationColumnValue Valor de la columna anterior para agrupar.
     * @return string
     */
    private function generateImaxTableRow($objectModelName, $id, $fields, $object = null, $hideColumns = array(), 
            $aggregationColumn = '', $aggregationColumnValue = '') {
        $this->imaxForm->generateImaxTableRow($objectModelName, $id, $fields, $object, $hideColumns, $aggregationColumn, $aggregationColumnValue);
        
        return $this;
    }
    
    /**
     * Graba automaticamente las tablas de ObjectModel.
     * @param array $post Debe contener imaxTableModels -> los ObjectModels cargados y imaxTableRow -> los datos correspondientes.
     * @param string $aggregationColumn Nombre de la columna por la que se agrupa.
     * @param string $aggregationColumnValue Valor de la columna anterior para agrupar.
     * @return boolean 
     */
    public static function saveFormObjectModelTables($post, $aggregationColumn = '', $aggregationColumnValue = false) {
        return imaxForm::saveFormObjectModelTables($post, $aggregationColumn, $aggregationColumnValue);
    }
}
