<?php
/**
 * Sirve para generar etiquetas.
 */
abstract class GeneradorEtiqueta {
    protected $errorEtiqueta = '', $modulo, $context;
    
    /**
     * 
     * @param ImaxImprimePedidosServidor $modulo
     */
    public function __construct($modulo) {
        $this->modulo = $modulo;
        $this->context = Context::getContext();
    }
    
    /**
     * Devuelve el identificador del transportista.
     * Se utiliza como nombre visible reemplazando los _ por espacios y poniéndole la primera letra en mayúsculas.
     * @return string
     */
    public abstract function obtenerId();

    /**
     * Genera la etiqueta correspondiente.
     * @param Order $pedido
     * @param boolean $marcarDescargado
     * @return string false en caso de error.
     */
    public function generarEtiqueta($pedido, $marcarDescargado) {
        $this->errorEtiqueta = '';
        $datosEtiqueta = $this->generarEtiquetaInterno($pedido, new BultosManuales($pedido->id));
        if ($datosEtiqueta) {
            if ($marcarDescargado) {
                $this->modulo->marcarDescargado($pedido->id, 'etiqueta');
            }
            
            $configuracion = $this->cargarConfiguracion();
            $this->modulo->cambiarEstado($pedido, $configuracion->nuevoEstado, true);
        }

        return $datosEtiqueta;
    }
    
    /**
     * Realiza la parte específica de generación de etiqueta.
     * @param Order $pedido
     * @param BultosManuales $bultosManuales
     * @return string false en caso de error.
     */
    protected abstract function generarEtiquetaInterno($pedido, $bultosManuales);
    
    /**
     * Devuelve el último error al generar una etiqueta.
     * Que devuelva algo no es indicativo de que exista error.
     * @return string
     */
    public function obtenerError() {
        return $this->errorEtiqueta;
    }
    
    /**
     * Guarda la configuración.
     * @param array $post
     * @return boolean
     */
    public function grabarConfiguracion($post) {
        $datos = $post['datosTransportista'][$this->obtenerId()];
        return Configuration::updateGlobalValue(ImaxImprimePedidosServidor::prefijo.'datosTransportista_'.$this->obtenerId(), serialize($datos));
    }
    
    /**
     * Devuelve la configuración.
     * @return ConfiguracionEtiqueta 
     */
    public function cargarConfiguracion() {
        $datos = Configuration::getGlobalValue(ImaxImprimePedidosServidor::prefijo.'datosTransportista_'.$this->obtenerId());
        if($datos) {
            $datos = unserialize($datos);
        }
        return new ConfiguracionEtiqueta($datos);
    }
    
    /**
     * Devuelve el html de la configuración.
     * @param array $transportistasAdaptados
     * @param array $estadosAdaptados
     * @return string
     */
    public function generarVisualizacion($transportistasAdaptados, $estadosAdaptados) {
        $configuracion = $this->cargarConfiguracion();
        $prefijo = 'datosTransportista['.$this->obtenerId().']';
        $nombre = ucfirst(str_replace('_', ' ', $this->obtenerId()));
        
        $acordeon = new imaxAcordeon($this->modulo->getPathUri());
        $formInterno = new imaxForm($this->modulo, $this->modulo->getPathUri(), '', '', '', '', false, true);
        $formInterno->createFormCheckboxGroupList($prefijo.'[idsReference]', $this->modulo->l("Selecciona los transportistas de $nombre:"), 
                $this->modulo->l("Selecciona los transportistas de $nombre:"), $transportistasAdaptados, $configuracion->idsReference);
        $formInterno->createFormSelect($prefijo.'[nuevoEstado]', $this->modulo->l("Estado de los pedidos de $nombre despues de generar la etiqueta:"), 
                $estadosAdaptados, $configuracion->nuevoEstado);
        $formInterno->createFormCheckboxGroup($prefijo.'[bultosManuales]', $this->modulo->l('Numero de bultos manual'), 
                $configuracion->bultosManuales);
        
        return $acordeon->renderAcordeon($nombre, $formInterno->renderForm(true), true);
    }
    
    /**
     * Devuelve el telefono falso para el pais de estar configurado.
     * @param Order $pedido
     * @return string
     */
    protected function obtenerTelefonoFalso($pedido) {
        return $this->modulo->obtenerTelefonoFalso($pedido);
    }
    
    /**
     * Devuelve la cantidad de paquetes por producto configurada en el modulo trzproductpackage, como minimo 1.
     * @param int $idProducto
     * @return int
     */
    protected function obtenerPaquetesTRZ($idProducto) {
        $idProducto = (int)$idProducto;
        
        try {
            $resultado = (int)Db::getInstance()->getValue('SELECT trzproductpackage FROM `'._DB_PREFIX_."trzproductpackageclass` WHERE id_product = '$idProducto'");
        }
        catch(Exception $ex) { }
        
        return (empty($resultado) ? 1 : $resultado);
    }
    
    /**
     * Devuelve un string con las referencias de los productos.
     * @param Order $pedido
     * @return string
     */
    protected function generarReferenciasString($pedido) {
        $detalles = $pedido->getProductsDetail();
        $referencias = array();
        foreach ($detalles AS $detalle) {
            $referencias[] = $detalle['product_reference'];
        }
        
        return ' REFs:' . implode(',', $referencias);
    }
    
    /**
     * Envia una solicitud.
     * @param string $url
     * @param array $post
     * @param string $method 
     * @param string[] $headers 
     * @return string
     */
    protected function sendCurl($url, $post, $method = 'post', $headers = []) {        
        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, TRUE);
        curl_setopt($ch, CURLOPT_BINARYTRANSFER, TRUE);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 0);
        curl_setopt($ch, CURLOPT_TIMEOUT, 360);
        if($headers) {
            curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        }
        if($method == 'post') {        
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $post);
        
        }       
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);
        $datos = curl_exec($ch);
        $infoCurl = curl_getinfo($ch);
        $numIntentos = 0;
        while (($infoCurl['http_code'] == 301 || $infoCurl['http_code'] == 302) && $numIntentos < 10) {
            curl_setopt($ch, CURLOPT_URL, $infoCurl['redirect_url']);
            $datos = curl_exec($ch);
            $infoCurl = curl_getinfo($ch);
            $numIntentos++;
        }

        curl_close($ch);

        return ($infoCurl['http_code'] == 200 && $numIntentos <= 10 && $datos ? $datos : false);
    }
    
    /**
     * Indica si la cantidad de bultos debe estar bloqueada.
     * @param Order $pedido
     * @return boolean
     */
    public function obtenerBultosBloqueados($pedido) {
        return false;
    }
}
