<?php

/**
 * 1. Líneas y botes iguales, todo ok.
 * 2. Si escaneas un bote del lote siguiente sin acabar el anterior. Puede ser que haya que marcar la linea anterior como merma o como 
 * "en pedido" si la encuentra.
 * 3. Menos líneas que botes, reservar las que haya disponibles y dar el aviso.
 */
class ConectorDeluxeTracking {
    private $intermediario;
    
    /**
     * Se ha reservado correctamente.
     */
    const RESERVA_CORRECTA = 0;
    /**
     * No hay numeros de lote suficientes.
     */
    const RESERVA_NO_LOCALIZADO = 1;
    /**
     * Se ha saltado algun numero de lote que deberia salir antes.
     */
    const RESERVA_INADECUADO = 2;
    /**
     * No se pudo ejecutar la reserva.
     */
    const RESERVA_NO_RESERVADO = 3;
    /**
     * El lote solicitado no existe o no es para el producto indicado.
     */
    const RESERVA_INEXISTENTE = 4;
    /**
     * Estados de la linea de tracking.
     */
    const ESTADO_INFORMACION = 0, ESTADO_DISPONIBLE = 1, ESTADO_MERMA = 2, ESTADO_VENDIDO = 3, ESTADO_EN_PEDIDO = 4;
    
    public function __construct() {
        $this->intermediario = new IntermediarioDeluxeTracking();
    }
    
    /**
     * Trata de reservar los lotes indicados y responde con una constante de reserva.
     * @param string $lote
     * @param int $id_product
     * @param int $id_product_attribute
     * @param int $cantidad
     * @param int $id_order
     * @param boolean $ignorarAnteriores Se ignora la comprobación de que el lote es el siguiente a usar.
     * @return array [int -> resultado, mixed -> extra]
     */
    public function reservarLote($lote, $id_product, $id_product_attribute, $cantidad, $id_order, $ignorarAnteriores = false) {
        //Deben coincidir en cantidad y numero de lote
        $siguiente = $this->intermediario->loteSiguiente($id_product, $id_product_attribute);
        if($siguiente) {
            $lotes = $this->intermediario->cargarLote($lote, $id_product, $id_product_attribute, $cantidad);
            if(!$lotes) {
                //Se ha escaneado algo que no esta disponible
                $esLote = $this->intermediario->esLote($lote, $id_product, $id_product_attribute);
                if($esLote) {
                    $siguiente = $this->intermediario->loteSiguiente($id_product, $id_product_attribute);
                    $resultado = [self::RESERVA_NO_LOCALIZADO, [$cantidad, $siguiente]];
                }
                else {
                    //No solo no esta disponible, no es para este producto o no existe
                    $resultado = [self::RESERVA_INEXISTENTE, null];
                }
            }
            elseif($siguiente == $lote || $ignorarAnteriores) {
                //Los reservamos
                $nombreProducto = Product::getProductName($id_product, $id_product_attribute);
                if($this->intermediario->modificarEstado($lote, $id_product, $id_product_attribute, count($lotes), self::ESTADO_EN_PEDIDO, 
                        $id_order, self::ESTADO_DISPONIBLE, $nombreProducto)) {
                    if(count($lotes) == $cantidad) {
                        //Es correcto
                        $resultado = [self::RESERVA_CORRECTA, null];
                    }
                    else {
                        //Se han reservado menos de los solicitados
                        $siguiente = $this->intermediario->loteSiguiente($id_product, $id_product_attribute);
                        $resultado = [self::RESERVA_NO_LOCALIZADO, [$cantidad - count($lotes), $siguiente]];
                    }
                }
                else {
                    //Error al realizar la reserva
                    $resultado = [self::RESERVA_NO_RESERVADO, null];
                }
            }
            else {
                //Se ha solicitado un lote que no es el correcto
                $loteSiguiente = $this->intermediario->cargarLote($siguiente, $id_product, $id_product_attribute, 0);
                $resultado = [self::RESERVA_INADECUADO, [$siguiente, count($loteSiguiente)]];
            }   
        }
        else {
            //No hay lote siguiente
            $resultado = [self::RESERVA_NO_LOCALIZADO, false];
        }
        
        return $resultado;
    }

    /**
     * Marca las lineas del lote como vendidas.
     * @param int $id_order
     * @param int $out_invoice
     * @return boolean
     */
    public function marcarVendido($id_order, $out_invoice = false) {
        return $this->intermediario->modificarEstadoPorPedido($id_order, self::ESTADO_VENDIDO, $out_invoice);
    }
    
    /**
     * Marca las lineas del lote como disponibles.
     * @param int $id_order
     * @return boolean
     */
    public function marcarDisponible($id_order) {
        return $this->intermediario->modificarEstadoPorPedido($id_order, self::ESTADO_DISPONIBLE, 0, 0);
    }
    
    /**
     * Marca las líneas como disponibles.
     * @param string $lote
     * @param int $id_product
     * @param int $id_product_attribute
     * @param int $cantidad
     * @param int $id_order
     * @return boolean
     */
    public function marcarDisponiblePorLote($lote, $id_product, $id_product_attribute, $cantidad, $id_order) {
        return $this->intermediario->marcarDisponible($lote, $id_product, $id_product_attribute, $cantidad, $id_order);
    }
    
    /**
     * Marca las lineas del lote como merma.
     * @param string $lote
     * @param int $id_product
     * @param int $id_product_attribute
     * @return array
     */
    public function marcarMerma($lote, $id_product, $id_product_attribute) {
        $mermas = [];
        
        $mermasActuales = $this->intermediario->cargarLote($lote, $id_product, $id_product_attribute, 0, self::ESTADO_MERMA);
        if($this->intermediario->modificarEstado($lote, $id_product, $id_product_attribute, 0, self::ESTADO_MERMA)) {
            $mermas = $this->intermediario->cargarLote($lote, $id_product, $id_product_attribute, 0, self::ESTADO_MERMA);
            //Quitamos las que ya estaban marcadas como merma
            $idsMermasActuales = [];
            foreach($mermasActuales as $mermaActual) {
                $idsMermasActuales[] = $mermaActual['id_tracking'];
            }
            foreach($mermas as $clave => $merma) {
                if(in_array($merma['id_tracking'], $idsMermasActuales)) {
                    unset($mermas[$clave]);
                }
            }
        }
        
        return $mermas;
    }
    
    /**
     * Devuelve los lotes en estado "en pedido" o "vendido" de un pedido ordenados por producto.
     * @param int $id_order
     * @return array
     */
    public function cargarReservados($id_order) {
        $reservadas = [];
        $lineas = $this->intermediario->cargarReservados($id_order);
        foreach($lineas as $linea) {
            $reservadas[(int)$linea['id_product']][(int)$linea['id_attribute']][] = $linea['out_lot'];
        }
        
        return $reservadas;
    }
    
    /**
     * Devuelve informacion sobre los lotes de un pedido.
     * @param int $id_order
     * @return array
     */
    public function cargarInfoPedido($id_order) {
        return $this->intermediario->cargarReservados($id_order);
    }
    
    /**
     * Indica si un producto aparece en la tabla de lotes.
     * @param int $id_product
     * @param int $id_product_attribute
     * @return boolean
     */
    public function usaLotes($id_product, $id_product_attribute) {
        return $this->intermediario->usaLotes($id_product, $id_product_attribute);
    }
}
